#!/usr/bin/env python3

# The code in this file is released under the Creative Commons CC0 license,
# i.e., for unrestricted use and without any warranty,
# see the file license_CC0.txt or https://creativecommons.org/publicdomain/zero/1.0/

from matplotlib.ticker import FormatStrFormatter
import matplotlib.pyplot as plt
import matplotlib as mpl
import numpy as np


def doPlotUgly(x_values, a_values, b_values):
    fig, ax = plt.subplots()
    mpl.rcParams.update({'font.size': 18})
    plt.ylim((0.5, 9.5))
    ax.plot(x_values, a_values, '-', label='Series A')
    ax.plot(x_values, b_values, '-', label='Series B')
    ax.xaxis.set_major_formatter(FormatStrFormatter('%.1e'))
    plt.xlabel('Time [s]')
    plt.ylabel('Lambda_ext [um]')
    ax.legend(loc='upper right', shadow=False)
    plt.savefig('ugly_example_plot.png', dpi=54)


def doPlotNice(x_values, a_values, b_values):
    fig, ax = plt.subplots()
    mpl.rcParams.update({'font.size': 18})
    x_values_min = x_values / 60  # convert seconds → minutes
    ax.plot(x_values_min, a_values, 'k-', lw=1.5, label='Series A')
    ax.plot(x_values_min, b_values, 'k--', lw=2, label='Series B')
    line_values =  [6 + np.pi for x in x_values]
    ax.plot(x_values_min, line_values, 'k:', lw=0.7)
    plt.xlabel('Time [min]')
    plt.xticks(np.arange(0, 90, 15))
    plt.ylabel('$\Lambda_{\\textnormal{ext}}$ [µm]')
    ax.legend(shadow=False)
    plt.legend(bbox_to_anchor=(0.88, 0.78), bbox_transform=plt.gcf().transFigure)

    a = plt.axes([.561, .2, .3, .3])
    plt.xlim((0, 5))
    plt.ylim((0, 6))
    a.plot(x_values_min, a_values, 'k-', lw=2)
    exp_values_upper = [6 + 2 * np.arctan(-24) + 3 * np.exp(-0.015*x) for x in x_values]
    a.plot(x_values_min, exp_values_upper, 'k:', lw=1)
    exp_values_lower = [6 + 2 * np.arctan(-24)  - 3 * np.exp(-0.015*x) for x in x_values]
    a.plot(x_values_min, exp_values_lower, 'k:', lw=1)
    plt.yticks((0,2,4,6))
    plt.savefig('nice_example_plot.pgf')


def fct_a(x):
    return \
        6 + \
        2 * np.arctan(0.01*(x-2400)) + \
        3 * np.sin(0.2*x) * np.exp(-0.015*x)


def fct_b(x):
    return \
        6 + np.pi/2 + \
        np.arctan(0.01*(x-2400))


if __name__ == '__main__':
    x_values = np.array(list(np.linspace(0, 199.9, 2000)) + list(np.linspace(200, 4800, 4601)))
    a_values = [fct_a(x) for x in x_values]
    b_values = [fct_b(x) for x in x_values]
    doPlotUgly(x_values, a_values, b_values)
    doPlotNice(x_values, a_values, b_values)
