/*
 * MIT License
 * Copyright (c) 2017 TeamViewer GmbH
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:

 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.

 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
*/

/**
  This example will connect to the API and will create a new sensor with the name "exampleSensor". Following it will create three metrics
  defined in the metric.c. There will be one metric defined over a valueUnit and two addtional metrics with user defined units.
  After the creation, these metrics will send values in the main loop. The gathering of the sensor values will be exemplary done in the
  sensorValue.c.

  You can stop the program by pressing CTRL + C or send a SIGTERM signal to this process.
 */
#if defined _WIN32 || defined _WIN64
#include <windows.h>
#endif

#include <stdio.h>
#include <signal.h>
#if defined _WIN32 || defined _WIN64
#include "unistd_win.h"
#else
#include <unistd.h>
#endif
#include <stdlib.h>

#include "tvcmi_client.h"

#include "userdata.h"
#include "client.h"
#include "metric.h"
#include "sensorValue.h"

static const char* ConfigFile = "./connector.conf";
/* The connector.conf file describes the connection information to the teamviewer-iot-agent.
 *
 * A possible content of that file is described in the following section:
 *
 * MqttHost             localhost
 * MqttProvisionPort    18883
 * MqttPort             18884
 * MqttKeepAlive        60
 * CaFile               /var/lib/teamviewer-iot-agent/certs/TeamViewerAuthority.crt
 * CertFolder           /home/pi/Downloads/certs
 *
 * Please note that all variables expect of the CertFolder has to be configured as it is shown. The CerFolder can be any folder
 * and it will be created if it is not already exists.
 */

static const char* nameOfSensor = "exampleSensor";
static int stopMainLoop = 0;

/*
 * @brief sig_handler Will handle signals send to your process, it will be used to stop the executable over SIGINT and SIGTERM
 * @param signo Signal Number send to the process.
 */
void SigHandler(int signo)
{
	if (signo == SIGTERM || signo == SIGINT)
	{
		printf("Will stop the main loop and exit the application\n");
		stopMainLoop = 1;
	}
}

int main(int argc, char* argv[])
{
	if (signal(SIGTERM, SigHandler) == SIG_ERR)
			printf("\ncan't catch SIGSTOP\n");
	if (signal(SIGINT, SigHandler) == SIG_ERR)
			printf("\ncan't catch SIGINT\n");

	TVCMI_LibInit();

	TVCMI_Error err;
	struct UserData userdata;
	userdata.readyToSendData = 0;
	userdata.isProvisioned = 0;
	userdata.nameOfSensor = nameOfSensor;

	userdata.client = TVCMI_NewClient(ConfigFile, &userdata, &err);

	//Provision the client
	Provision(userdata.client);

	//Wait untill the client is provisioned.
	while (userdata.isProvisioned == 0)
	{
		sleep(1);
	}

	Connect(userdata.client);

	while (!stopMainLoop)
	{
		if (userdata.readyToSendData == 1)
		{
			const double temp = GetTemperatureData();
			const int productivity = GetProductivityData();
			char buffer[50];
			GetCurrentProducedGood(buffer, 50);
			printf("Push Metric Values with Temperature [%f], Productivity [%i] and Product [%s] \n", temp, productivity, buffer);
			PushValues(&userdata, temp, productivity, buffer);
		}
		sleep(3);
	}

	free((void*)userdata.sensorId);
	TVCMI_DestroyCreatedMetrics(userdata.createdMetrics);
	TVCMI_Disconnect(userdata.client);
	TVCMI_DeleteClient(userdata.client);
	TVCMI_LibCleanup();

	return 0;
}
