/*
 * MIT License
 * Copyright (c) 2017 TeamViewer GmbH
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:

 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.

 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
*/

#include <stdio.h>
#include <string.h>
#if defined _WIN32 || defined _WIN64
#include "unistd_win.h"
#else
#include <unistd.h>
#endif
#include <signal.h>
#include <stdlib.h>
#include <string.h>

#include "metric.h"


static const char* tempMetricName = "Indoor Temperature";
static const char* tempMetricMatchingId = "1";

static const char* productivityMetricName = "Productivity Machine 1";
static const char* productivityMetricMatchingId = "2";
static const char* productivityMetricValueAnnotation = "Goods per minute";

static const char* productMetricName = "Product";
static const char* productMetricMatchingId = "3";
static const char* productMetricValueAnnotation = "Current produced product";


void CreateMetrics(TVCMI_Client* client, struct UserData* userData)
{
	TVCMI_CreateMetricsCallbackSet(client, &OnCreateMetrics);

	TVCMI_MetricSchemaContainer* schemaContainer = TVCMI_CreateMetricSchemaContainer();
	TVCMI_AddStandardMetricSchema(schemaContainer, tempMetricMatchingId, tempMetricName, TVCMI_ValueUnit_SICelsius);
	TVCMI_AddCustomMetricSchema(schemaContainer,productivityMetricMatchingId,productivityMetricName, TVCMI_ValueType_Integer, productivityMetricValueAnnotation);
	TVCMI_AddCustomMetricSchema(schemaContainer,productMetricMatchingId,productMetricName, TVCMI_ValueType_String, productMetricValueAnnotation);

	TVCMI_CreateMetrics(client, userData->sensorId, schemaContainer);
	TVCMI_DestroyMetricSchemaContainer(schemaContainer);
}

void OnPushMetricValues(TVCMI_Client* client, void* userData, const char* sensorId, TVCMI_ApiError err)
{
	struct UserData* data = (struct UserData*) userData;
}

void PushValues(struct UserData* data, const double tempValue, const int productivity, const char* currentProducedGoods)
{
	TVCMI_MetricValueContainer* valueContainer = TVCMI_CreateMetricValueContainer();

	TVCMI_AddDoubleMetricValue(valueContainer, tempValue, data->tempId);
	TVCMI_AddIntegerMetricValue(valueContainer, productivity, data->productivityId);
	TVCMI_AddStringMetricValue(valueContainer, currentProducedGoods, data->productId);

	TVCMI_PushMetricValues(data->client, data->sensorId, valueContainer);
	TVCMI_DestroyMetricValueContainer(valueContainer);
}

void OnCreateMetrics(TVCMI_Client* client, void* userData, TVCMI_CreatedMetrics* createdMetrics, TVCMI_ApiError err)
{
	struct UserData* data = (struct UserData*) userData;
	data->createdMetrics = createdMetrics;
	TVCMI_CreatedMetric* metrics = createdMetrics->metrics;

	for (uint8_t position = 0; position < createdMetrics->metricsCount; ++position)
	{
		TVCMI_CreatedMetric metric = metrics[position];

		if (strncmp(metric.matchingId,tempMetricMatchingId,1) == 0)
		{
			data->tempId = metric.metricId;
			printf("Created metricId for [%s] is %s \n", tempMetricName, data->tempId);
		}
		else if (strncmp(metric.matchingId,productivityMetricMatchingId,1) == 0)
		{
			data->productivityId = metric.metricId;
			printf("Created metricId for [%s] is %s \n",productivityMetricName, data->productivityId);
		}
		else if (strncmp(metric.matchingId,productMetricMatchingId,1) == 0)
		{
			data->productId = metric.metricId;
			printf("Created metricId for [%s] is %s \n",productMetricName, data->productId);
		}
	}

	TVCMI_PushMetricValuesCallbackSet(data->client, &OnPushMetricValues);

	//activate pushing of metric values in main loop
	data->readyToSendData = 1;
}
