/*
 * MIT License
 * Copyright (c) 2017 TeamViewer GmbH
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:

 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.

 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
*/

/**
  This example will connect to the API and will print out the current inventory of already created sensors and metrics in the format of:

Sensors of client
         Sensor name:    exampleSensor
         Sensor Id:      31a16f9d67bd4671b43ff8a0ad09d863
                 Metric Id: 2a9a5bc001b948f8b54d3dafaca619e9
                 Metric Id: 31fc1d88e0a44a688555cda8d56f2316
                 Metric Id: cfd437b72f904645a5f9e04b8590e7dc

	For detailed information of a particular metric consider the function TVCMI_GetMetricInventory from the SDK.
	After the printing it will terminate.
 */
#if defined _WIN32 || defined _WIN64
#include <windows.h>
#endif

#include <stdio.h>
#include <signal.h>
#if defined _WIN32 || defined _WIN64
#include "unistd_win.h"
#else
#include <unistd.h>
#endif
#include <stdlib.h>

#include "tvcmi_client.h"

#include "client.h"
#include "userdata.h"

static const char* ConfigFile = "./connector.conf";
/* The connector.conf file describes the connection information to the teamviewer-iot-agent.
 *
 * A possible content of that file is described in the following section:
 *
 * MqttHost             localhost
 * MqttProvisionPort    18883
 * MqttPort             18884
 * MqttKeepAlive        60
 * CaFile               /var/lib/teamviewer-iot-agent/certs/TeamViewerAuthority.crt
 * CertFolder           /home/pi/Downloads/certs
 *
 * Please note that all variables expect of the CertFolder has to be configured as it is shown. The CerFolder can be any folder
 * and it will be created if it is not already exists.
 */

static int stopMainLoop = 0;

/**
 * @brief sig_handler Will handle signals send to your process, it will be used to stop the executable over SIGINT and SIGTERM
 * @param signo Signal Number send to the process.
 */
void SigHandler(int signo)
{
	if (signo == SIGTERM || signo == SIGINT)
	{
		printf("Will stop the main loop and exit the application\n");
		stopMainLoop = 1;
	}
}

int main(int argc, char* argv[])
{
	if (signal(SIGTERM, SigHandler) == SIG_ERR)
			printf("\ncan't catch SIGSTOP\n");
	if (signal(SIGINT, SigHandler) == SIG_ERR)
			printf("\ncan't catch SIGINT\n");

	TVCMI_LibInit();

	TVCMI_Error err;
	struct UserData userdata;
	userdata.isProvisioned = 0;

	userdata.client = TVCMI_NewClient(ConfigFile, &userdata, &err);

	//Provision the client
	Provision(userdata.client);

	//Wait untill the client is provisioned.
	while (userdata.isProvisioned == 0)
	{
		sleep(1);
	}

	Connect(userdata.client);

	while (stopMainLoop == 0)
	{
		sleep(3);
	}

	TVCMI_Disconnect(userdata.client);
	TVCMI_DeleteClient(userdata.client);
	TVCMI_LibCleanup();

	return 0;
}
