/*
 * MIT License
 * Copyright (c) 2017 TeamViewer GmbH
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
*/

///
/// @file tvcmi_client.h
/// tvcmiclient - client library for the TeamViewer IOT Agent API version 1.0
///
///
/// Usage hints
///
/// Call TVCMI_LibInit() function before using any other functions of this library.
///
/// All callback functions should be set once. Otherwise they will be replaced and that
/// will not be thread safe.
///


#ifndef __TVCMI_CLIENT_H__
#define __TVCMI_CLIENT_H__

#include <stdint.h>
#include <time.h>

#ifdef __cplusplus
extern "C" {
#endif

#define IOTTVCMI_API
#if defined(_MSC_VER)
#	if defined(IOTTVCMI_CREATE_SHARED_LIBRARY)
#		undef IOTTVCMI_API
#		define IOTTVCMI_API __declspec(dllexport)
#	elif defined(IOTTVCMI_LINKED_AS_SHARED_LIBRARY)
#		undef IOTTVCMI_API
#		define IOTTVCMI_API __declspec(dllimport)
#	endif
#endif

/// The version of the TeamViewer IOT Agent API.
#define ApiVersion "v1.0"


/// Gateway errors.
typedef enum TVCMI_ApiErrorTag
{
    TVCMI_ApiError_NoError = -1,
    TVCMI_ApiError_UnknownError = 0,
    TVCMI_ApiError_NoBackendConnection = 1,
    TVCMI_ApiError_NoLicence = 2,
    TVCMI_ApiError_InternalError = 3,
    TVCMI_ApiError_InvalidJson = 4,
    TVCMI_ApiError_JsonIsNotAnObject = 5,
    TVCMI_ApiError_MissingParameter = 6,
    TVCMI_ApiError_ParameterIsNotAString = 7,
    TVCMI_ApiError_ParameterIsNotANumber = 8,
    TVCMI_ApiError_ParameterIsNotABool = 9,
    TVCMI_ApiError_ParameterIsNotAnArray = 10,
    TVCMI_ApiError_ParameterIsNotAnInteger = 11,
    TVCMI_ApiError_UnknownValueUnit = 12,
    TVCMI_ApiError_UnknownMetricType = 13,
    TVCMI_ApiError_ValueUnitOrValueTypeHasToBeProvided = 14,
    TVCMI_ApiError_UnknownSensorId = 15,
    TVCMI_ApiError_UnknownMetricId = 16,
    TVCMI_ApiError_NoMetricsUpdated = 17,
    TVCMI_ApiError_ForbiddendAccess = 18,
    TVCMI_ApiError_OutOfRange = 19,
    TVCMI_ApiError_CertificateFormatError = 20,
    TVCMI_ApiError_CertificateIsEmptyError = 21,
    TVCMI_ApiError_CertificateWithoutCommonName = 22,
    TVCMI_ApiError_DeleteMetricsFailed = 23,
    TVCMI_ApiError_ChangeMetricsFailed = 24,
    TVCMI_ApiError_InvalidClientName = 25,
    TVCMI_ApiError_CreateMetricsFailed = 26,
    TVCMI_ApiError_PushValuesFailed = 27

} TVCMI_ApiError;

/// Error values.
typedef enum TVCMI_ErrorTag
{
    TVCMI_Error_Unknown = -1,
    TVCMI_Error_Success = 0,
    TVCMI_Error_WrongApiVersion = 1,
    TVCMI_Error_WrongConfigName = 2,
    TVCMI_Error_ConfigReadFailed = 3,
    TVCMI_Error_WrongConfigParams = 4,
    TVCMI_Error_NoMemory = 5,
    TVCMI_Error_InvalidArguments = 6,
    TVCMI_Error_NoResource = 7,
    TVCMI_Error_ConnectionFailed = 8,
    TVCMI_Error_WrongProtocol = 9,
    TVCMI_Error_WrongIdentifier = 10,
    TVCMI_Error_ServerUnavailable = 11,
    TVCMI_Error_UnexpectedDisconnect = 12,
    TVCMI_Error_NoConnectionToServer = 13,
    TVCMI_Error_PayloadSize = 14,
    TVCMI_Error_WrongReply = 15,
    TVCMI_Error_Busy = 16,
    TVCMI_Error_CreateCertificate = 17,
    TVCMI_Error_CreateProvisionCertificate = 18,
    TVCMI_Error_CreatePrivateKey = 19,
    TVCMI_Error_GetClientId = 20,
    TVCMI_Error_TlsSetFailed = 21,
    TVCMI_Error_ClientProvision = 22,
    TVCMI_Error_ModifySensor = 23,
    TVCMI_Error_DeleteSensor = 24,
    TVCMI_Error_DeleteMetrics = 25,
    TVCMI_Error_RemoveCertificates = 26,
    TVCMI_Error_TlsConnection = 27,
    TVCMI_Error_Errno = 28,
    TVCMI_Error_ComposeJson = 29,
    TVCMI_Error_NoThreadSupport = 30,
    TVCMI_Error_ReadCertificateCN = 31,
    TVCMI_Error_CertificateRequestPath = 32,
    TVCMI_Error_CertifcateFolderPath = 33,
    TVCMI_Error_AlreadyProvisioned = 34,
    TVCMI_Error_WritingToCertificateFile = 35,
    TVCMI_Error_WrongCertificateFile = 36,
    TVCMI_Error_ThreadCreate = 37,
    TVCMI_Error_NoBackendConnection = 38,
    TVCMI_Error_CertificateRequest = 39

} TVCMI_Error;

/// Metric value units.
typedef enum TVCMI_ValueUnitTag
{
    TVCMI_ValueUnit_NoUnit,
    TVCMI_ValueUnit_SIAmpere,
    TVCMI_ValueUnit_SIBit,
    TVCMI_ValueUnit_SICandela,
    TVCMI_ValueUnit_SICelsius,
    TVCMI_ValueUnit_SIFarad,
    TVCMI_ValueUnit_SIKilogram,
    TVCMI_ValueUnit_SIHertz,
    TVCMI_ValueUnit_SIJoule,
    TVCMI_ValueUnit_SIMeter,
    TVCMI_ValueUnit_SIMetersPerSecond,
    TVCMI_ValueUnit_SIMetersPerSquareSecond,
    TVCMI_ValueUnit_SIMole,
    TVCMI_ValueUnit_SINewtone,
    TVCMI_ValueUnit_SIOhm,
    TVCMI_ValueUnit_SIPascal,
    TVCMI_ValueUnit_SIRadian,
    TVCMI_ValueUnit_SISecond,
    TVCMI_ValueUnit_SISquareMeter,
    TVCMI_ValueUnit_SIVolt,
    TVCMI_ValueUnit_SIWatt,
    TVCMI_ValueUnit_NoSIDecibel,  // Non-SI Decibel unit
    TVCMI_ValueUnit_NoSIPercent   // Non-SI Percent unit

} TVCMI_ValueUnit;

/// Metric value types.
typedef enum TVCMI_ValueTypeTag
{
    TVCMI_ValueType_NoType,
    TVCMI_ValueType_Integer,
    TVCMI_ValueType_Double,
    TVCMI_ValueType_Bool,
    TVCMI_ValueType_String

} TVCMI_ValueType;

/// Created metric information. Used in TVCMI_CreatedMetrics.
typedef struct TVCMI_CreatedMetricTag
{
    char* matchingId;
    char* metricId;

} TVCMI_CreatedMetric;

/// Created metrics information.
typedef struct TVCMI_CreatedMetricsTag
{
    char*                    sensorId;
    TVCMI_CreatedMetric*     metrics;
    int                      metricsCount;

} TVCMI_CreatedMetrics;

/// Metric information. Used in TVCMI_SensorInfo.
typedef struct TVCMI_MetricInfoTag
{
    char* metricId;

} TVCMI_MetricInfo;

/// Sensor types
typedef enum TVCMI_SensorTypeTag
{
    TVCMI_SensorType_None,
    TVCMI_SensorType_Standard,
    TVCMI_SensorType_Plugin

} TVCMI_SensorType;

/// Plugin parameters information. Used in TVCMI_SensorMetadata.
typedef struct TVCMI_PluginParameterTag
{
    char*               parameterName;
    char*               parameterValue;

} TVCMI_PluginParameter;

/// Sensor information. Used in TVCMI_SensorsInfo.
typedef struct TVCMI_SensorMetadataTag
{
    char*                   name;
    char*                   sensorId;
    int                     frequency;
    TVCMI_SensorType        sensorType;
    char*                   pluginName;
    char*                   pluginDownloadUrl;
    TVCMI_PluginParameter*  pluginParameters;
    int                     pluginParametersCount;
    char*                   monitoringService;
    TVCMI_MetricInfo*       metrics;
    int                     metricsCount;

} TVCMI_SensorMetadata;

/// Sensors information.
typedef struct TVCMI_SensorsInfoTag
{
    TVCMI_SensorMetadata*   sensors;
    int                     sensorsCount;

} TVCMI_SensorsInfo;

/// Metric information.
typedef struct TVCMI_MetricMetadataTag
{
    char*           id;
    TVCMI_ValueUnit valueUnit;
    TVCMI_ValueType valueType;
    char*           valueAnnotation;
    uint64_t        lastUpdated;
    char*           value;
    char*           name;

} TVCMI_MetricMetadata;

/// A container for creating metrics.
typedef struct TVCMI_MetricSchemaContainerTag TVCMI_MetricSchemaContainer;

/// A container for sending metric values.
typedef struct TVCMI_MetricValueContainerTag TVCMI_MetricValueContainer;

/// A client object.
typedef struct TVCMI_ClientTag TVCMI_Client;


///
/// Initializes the library.
/// It must be called before any other API functions. This have to be called only once in the beginning.
/// This function is not thread safe.
///
/// @return TVCMI_Error_Success -   always
///
/// @see TVCMI_LibCleanup()
///
IOTTVCMI_API TVCMI_Error  TVCMI_LibInit();


///
/// Call TVCMI_LibCleanup to free resources associated with the library.
/// This function is not thread safe.
///
/// @return TVCMI_Error_Success -   always
///
/// @see TVCMI_LibInit()
///
IOTTVCMI_API TVCMI_Error TVCMI_LibCleanup();


///
/// Create a client object.
/// It must be deleted by TVCMI_DeleteClient() after usage.
///
/// @param confFile          A file path which will contain all the API protocol
///                          specific configuration data. A possible content of that file
///                          is described in the following section:
///                          - MqttHost             localhost // host to connect
///                          - MqttProvisionPort    1883 // port for the client
///                                                         provisioning
///                          - MqttPort             8883 // port for other than
///                                                         provisioning actions
///                          - MqttKeepAlive        60 // connection keep alive seconds
///                          - CaFile               /var/lib/teamviewer-iot-agent/certs/TeamViewerAuthority.crt
///                                                     // Team Viewer CA certificate path
///                          - CertFolder           /home/pi/Downloads/certs
///                                                     // a folder where the certificates will be placed and it will be created if
///                                                     it does not already exist. Each client must have its own folder for certificates.
///                          - ClientName           IOT-Client // a display name for the client
/// @param callbackParameter                    A pointer that will be passed as an
///                                             argument to any callbacks function.
/// @param err               Returned error code.
///
/// @return TVCMI_Error_Success -                       on success
/// @return TVCMI_Error_InvalidArguments -              wrong parameters
/// @return TVCMI_Error_NoMemory -                      no memory
/// @return TVCMI_Error_WrongConfigName -               confFile argument is NULL or empty
/// @return TVCMI_Error_ConfigReadFailed -              can not open the file confFile
/// @return TVCMI_Error_WrongConfigParams -             the file confFile has wrong parameters in it
///
IOTTVCMI_API TVCMI_Client* TVCMI_NewClient(const char* confFile, void* callbackParameter, TVCMI_Error* err);


///
/// Use to free the memory associated with a TVCMI_Client instance.
/// Do not call this function from the callback functions.
///
/// @param client    A struct TVCMI_Client pointer to free.
///
/// @return TVCMI_Error_Success -           on success
/// @return TVCMI_Error_InvalidArguments -  wrong parameters
///
IOTTVCMI_API TVCMI_Error TVCMI_DeleteClient(TVCMI_Client* client);


///
/// Provision the client. It will provision the client, if it is not provisioned yet.
/// The callback function set by TVCMI_ProvisionClientCallbackSet will inform about the
/// client provisioning on success or will return an error code in case of failure.
///
/// @param client               A valid TVCMI_Client instance.
///
/// @return TVCMI_Error_Success -                       on success
/// @return TVCMI_Error_InvalidArguments -              wrong parameters
/// @return TVCMI_Error_NoMemory -                      no memory
/// @return TVCMI_Error_GetClientId -                   failed to get client id from
///                                                     certificate
/// @return TVCMI_Error_CreateProvisionCertificate -    failed to create provision
///                                                     certificate
/// @return TVCMI_Error_CreatePrivateKey -              failed to create a private key
/// @return TVCMI_Error_AlreadyProvisioned -            the client was provisioned
/// @return TVCMI_Error_TlsSetFailed -                  failed to set TLS options
/// @return TVCMI_Error_NoThreadSupport -               the system does not support threads
/// @return TVCMI_Error_Errno -                         system error number
///
IOTTVCMI_API TVCMI_Error TVCMI_ProvisionClient(TVCMI_Client* client);


///
/// Connect to the TeamViewer IOT Agent. Callback function set by TVCMI_ConnectCallbackSet() will
/// inform about connection done.
///
/// @param client               A valid TVCMI_Client instance.
///
/// @return TVCMI_Error_Success -           on success
/// @return TVCMI_Error_NoResource -        no resources
/// @return TVCMI_Error_InvalidArguments -  wrong parameters
/// @return TVCMI_Error_TlsSetFailed -      failed to set TLS options
/// @return TVCMI_Error_NoThreadSupport -   the system does not support threads
/// @return TVCMI_Error_Errno -             system error number
///
IOTTVCMI_API TVCMI_Error TVCMI_Connect(TVCMI_Client* client);


///
/// Disconnect from the TeamViewer IOT Agent. Callback function set by
/// TVCMI_DisconnectCallbackSet() will inform about
/// connection drop.
///
/// @param client               A valid TVCMI_Client instance.
///
/// @return TVCMI_Error_Success -           on success
/// @return TVCMI_Error_InvalidArguments -  wrong parameters
///
IOTTVCMI_API TVCMI_Error TVCMI_Disconnect(TVCMI_Client* client);


///
/// Create sensor by name. Callback function set by TVCMI_CreteSensorCallbackSet()
/// will return a sensor id.
///
/// @param client               A valid TVCMI_Client instance.
/// @param name                 Sensor name.
///
/// @return TVCMI_Error_Success -               on success
/// @return TVCMI_Error_InvalidArguments -      wrong parameters
/// @return TVCMI_Error_NoMemory -              out of memory
/// @return TVCMI_Error_NoConnectionToServer -  no connection with TeamViewer IOT Agent
/// @return TVCMI_Error_WrongProtocol -         internal protocol error communicating with
///                                             the TeamViewer IOT Agent
///
IOTTVCMI_API TVCMI_Error TVCMI_CreateSensor(TVCMI_Client* client, const char* name);


///
/// Update sensor meta data. The callback function
/// set by TVCMI_UpdateSensorCallbackSet() will return success or
/// error information about sensor meta data change.
///
/// @param client           A valid TVCMI_Client instance.
/// @param sensorId         ID of a sensor.
/// @param sensorName       Sensor name. If NULL, it will not be changed.
///
/// @return TVCMI_Error_Success -               on success
/// @return TVCMI_Error_InvalidArguments -      wrong parameters
/// @return TVCMI_Error_NoMemory -              out of memory
/// @return TVCMI_Error_NoConnectionToServer -  no connection with TeamViewer IOT Agent
/// @return TVCMI_Error_WrongProtocol -         internal protocol error communicating with
///                                             the TeamViewer IOT Agent
///
IOTTVCMI_API TVCMI_Error TVCMI_UpdateSensor(TVCMI_Client* client, const char* sensorId, const char* sensorName);


///
/// Delete sensor. The callback function
/// set by TVCMI_DeleteSensorCallbackSet() will return success or
/// error information about sensor deletion.
///
/// @param client           A valid TVCMI_Client instance.
/// @param sensorId         ID of a sensor to delete.
///
/// @return TVCMI_Error_Success -               on success
/// @return TVCMI_Error_InvalidArguments -      wrong parameters
/// @return TVCMI_Error_NoMemory -              out of memory
/// @return TVCMI_Error_NoConnectionToServer -  no connection with TeamViewer IOT Agent
/// @return TVCMI_Error_WrongProtocol -         internal protocol error communicating with
///                                             the TeamViewer IOT Agent
///
IOTTVCMI_API TVCMI_Error TVCMI_DeleteSensor(TVCMI_Client* client, const char* sensorId);


///
/// Create metrics of a sensor. The callback function set by
/// TVCMI_CreateMetricsCallbackSet() will return created
/// metric IDs on success or error code on failure.
///
/// @param client               A valid TVCMI_Client instance.
/// @param sensorId             ID of a sensor.
/// @param container            A structure containing metric schemas.
///
/// @return TVCMI_Error_Success -               on success
/// @return TVCMI_Error_InvalidArguments -      wrong parameters
/// @return TVCMI_Error_NoMemory -              out of memory
/// @return TVCMI_Error_NoConnectionToServer -  no connection with TeamViewer IOT Agent
/// @return TVCMI_Error_WrongProtocol -         internal protocol error communicating with
///                                             the TeamViewer IOT Agent
/// @return TVCMI_Error_ComposeJson -           could not create json message
///
/// @see TVCMI_CreateMetricSchemaContainer()
/// @see TVCMI_DestroyMetricSchemaContainer()
/// @see TVCMI_AddStandardMetricSchema()
/// @see TVCMI_AddCustomMetricSchema()
///
IOTTVCMI_API TVCMI_Error TVCMI_CreateMetrics(TVCMI_Client* client, const char* sensorId, TVCMI_MetricSchemaContainer* container);


///
/// Update metrics. The callback function set by
/// TVCMI_UpdateMetricsCallbackSet() will return success or
/// error information about metrics update.
///
/// @param client               A valid TVCMI_Client instance.
/// @param sensorId             ID of a sensor.
/// @param container            A structure containing metric schemas.
///
/// @return TVCMI_Error_Success -               on success
/// @return TVCMI_Error_InvalidArguments -      wrong parameters
/// @return TVCMI_Error_NoMemory -              out of memory
/// @return TVCMI_Error_NoConnectionToServer -  no connection with TeamViewer IOT Agent
/// @return TVCMI_Error_WrongProtocol -         internal protocol error communicating with
///                                             the TeamViewer IOT Agent
/// @return TVCMI_Error_ComposeJson -           could not create json message
///
/// @see TVCMI_CreateMetricSchemaContainer()
/// @see TVCMI_AddMetricUpdateSchema()
/// @see TVCMI_DestroyMetricSchemaContainer()
///
IOTTVCMI_API TVCMI_Error TVCMI_UpdateMetrics(TVCMI_Client* client, const char* sensorId, TVCMI_MetricSchemaContainer* container);


///
/// Send values of metrics of a client to the TeamViewer IOT Agent.
/// If the sensor or metrics were not registered, then they will be registered before pushing values.
/// The callback function set by TVCMI_PushMetricValuesCallbackSet() will return error codes in
/// case of failure.
///
/// @param client           A valid TVCMI_Client instance.
/// @param sensorId         ID of a sensor
/// @param container        A pointer to a structure containing metric values.
///
/// @return TVCMI_Error_Success -               on success
/// @return TVCMI_Error_InvalidArguments -      wrong parameters
/// @return TVCMI_Error_NoMemory -              out of memory
/// @return TVCMI_Error_NoConnectionToServer -  no connection with TeamViewer IOT Agent
/// @return TVCMI_Error_WrongProtocol -         internal protocol error communicating with
///                                             the TeamViewer IOT Agent
///
/// @see TVCMI_CreateMetricValueContainer()
/// @see TVCMI_DestroyMetricValueContainer()
/// @see TVCMI_AddIntegerMetricValue()
/// @see TVCMI_AddBoolMetricValue()
/// @see TVCMI_AddDoubleMetricValue()
/// @see TVCMI_AddStringMetricValue()
/// @see TVCMI_AddEpochTimeStamp()
/// @see TVCMI_PushMetricValuesCallbackSet()
///
IOTTVCMI_API TVCMI_Error TVCMI_PushMetricValues(TVCMI_Client* client, const char* sensorId, TVCMI_MetricValueContainer* container);


///
/// Delete metrics.
///
/// @param client           A valid TVCMI_Client instance.
/// @param sensorId         ID of a sensor.
/// @param metricIds        An array of metric IDs.
/// @param metricsCount     The number of metrics in the metricIds array.
///
/// @return TVCMI_Error_Success -               on success
/// @return TVCMI_Error_InvalidArguments -      wrong parameters
/// @return TVCMI_Error_NoMemory -              out of memory
/// @return TVCMI_Error_NoConnectionToServer -  no connection with TeamViewer IOT Agent
/// @return TVCMI_Error_WrongProtocol -         internal protocol error communicating with
///                                             the TeamViewer IOT Agent
///
IOTTVCMI_API TVCMI_Error TVCMI_DeleteMetrics(TVCMI_Client* client, const char* sensorId, char** metricIds, int metricsCount);


///
/// Create a TVCMI_MetricSchemaContainer object and returns it's pointer. It should be
/// destroyed by function TVCMI_DestroyMetricSchemaContainer().
///
/// @return TVCMI_MetricSchemaContainer* -  on success
/// @return NULL -                          on failure
///
/// @see TVCMI_DestroyMetricSchemaContainer()
/// @see TVCMI_AddStandardMetricSchema()
/// @see TVCMI_AddCustomMetricSchema()
///
IOTTVCMI_API TVCMI_MetricSchemaContainer* TVCMI_CreateMetricSchemaContainer();


///
/// Destroy container object created by function
/// TVCMI_CreateMetricSchemaContainer().
///
/// @param container          A valid TVCMI_MetricSchemaContainer
///                           pointer to destroy.
///
/// @return TVCMI_Error_Success -           on success
/// @return VCMI_Error_InvalidArguments -   wrong parameters
///
/// @see TVCMI_CreateMetricSchemaContainer()
/// @see TVCMI_AddStandardMetricSchema()
/// @see TVCMI_AddCustomMetricSchema()
///
IOTTVCMI_API TVCMI_Error TVCMI_DestroyMetricSchemaContainer(TVCMI_MetricSchemaContainer* container);


///
/// Add a metric with standard ValueUnit type of value unit.
///
/// @param container            A valid TVCMI_MetricValueContainer pointer.
/// @param matchingId           This string will be used to identify the metric ID in the
///                             response. It has to be unique in the context of this
///                             request.
/// @param name                 Metric name.
/// @param valueUnit            Value unit of metric.
///
/// @return TVCMI_Error_Success -           on success
/// @return TVCMI_Error_NoMemory -          no memory
/// @return TVCMI_Error_InvalidArguments -  wrong parameters
/// @return TVCMI_Error_ComposeJson -       could not create json message
///
/// @see TVCMI_CreateMetricSchemaContainer()
/// @see TVCMI_DestroyMetricSchemaContainer()
/// @see TVCMI_AddCustomMetricSchema()
///
IOTTVCMI_API TVCMI_Error TVCMI_AddStandardMetricSchema(
	TVCMI_MetricSchemaContainer* container,
	const char* matchingId,
	const char* name,
	TVCMI_ValueUnit valueUnit);


///
/// Add a metric with custom type of value unit.
///
/// @param container        A valid TVCMI_MetricValueContainer
///                         pointer.
/// @param matchingId       An ID which is unique in
///                         sensor and can be used to match
///                         metric IDs after registration.
/// @param name             Metric name.
/// @param valueType        Custom value type of metric.
/// @param valueAnnotation  Custom measurement unit ("kg", "Mhz",
///                         "Km", "", ...).
///
/// @return TVCMI_Error_Success -           on success
/// @return TVCMI_Error_NoMemory -          no memory
/// @return TVCMI_Error_InvalidArguments -  wrong parameters
/// @return TVCMI_Error_ComposeJson -       could not create json message
///
/// @see TVCMI_CreateMetricSchemaContainer()
/// @see TVCMI_DestroyMetricSchemaContainer()
/// @see TVCMI_AddCustomMetricSchema()
///
IOTTVCMI_API TVCMI_Error TVCMI_AddCustomMetricSchema(
	TVCMI_MetricSchemaContainer* container,
	const char* matchingId,
	const char* name,
	TVCMI_ValueType valueType,
	const char* valueAnnotation);


///
/// Add metrics to change name and/or valueAnnotation.
///
/// @param container        A valid TVCMI_MetricValueContainer
///                         pointer.
/// @param metricId         metric ID.
/// @param name             Metric name.
/// @param valueAnnotation  Custom measurement unit ("kg", "Mhz",
///                         "Km", "", ...).
///
/// @return TVCMI_Error_Success -           on success
/// @return TVCMI_Error_NoMemory -          no memory
/// @return TVCMI_Error_InvalidArguments -  wrong parameters
/// @return TVCMI_Error_ComposeJson -       could not create json message
///
/// @see TVCMI_CreateMetricSchemaContainer()
/// @see TVCMI_DestroyMetricSchemaContainer()
///
IOTTVCMI_API TVCMI_Error TVCMI_AddMetricUpdateSchema(
	TVCMI_MetricSchemaContainer* container,
	const char* metricId,
	const char* name,
	const char* valueAnnotation);


///
/// Create a TVCMI_MetricValueContainer object. Should be destroyed by function
/// TVCMI_DestroyMetricValueContainer().
///
/// @return TVCMI_MetricValueContainer* -    on success
/// @return NULL -                           on failure
///
/// @see TVCMI_DestroyMetricValueContainer()
/// @see TVCMI_AddIntegerMetricValue()
/// @see TVCMI_AddBoolMetricValue()
/// @see TVCMI_AddDoubleMetricValue()
/// @see TVCMI_AddStringMetricValue()
/// @see TVCMI_AddEpochTimeStamp()
///
IOTTVCMI_API TVCMI_MetricValueContainer* TVCMI_CreateMetricValueContainer();


///
/// Destroy container object created by function
/// TVCMI_CreateMetricValueContainer().
///
/// @param container        A valid TVCMI_MetricValueContainer pointer
///                         to destroy.
///
/// @return TVCMI_Error_Success -          on success
/// @return TVCMI_Error_InvalidArguments - wrong parameters
///
/// @see TVCMI_CreateMetricValueContainer()
/// @see TVCMI_AddIntegerMetricValue()
/// @see TVCMI_AddBoolMetricValue()
/// @see TVCMI_AddDoubleMetricValue()
/// @see TVCMI_AddStringMetricValue()
/// @see TVCMI_AddEpochTimeStamp()
///
IOTTVCMI_API TVCMI_Error TVCMI_DestroyMetricValueContainer(TVCMI_MetricValueContainer* container);


///
/// Add a value of type int64_t to the container object.
///
/// @param container        A valid TVCMI_MetricValueContainer object.
/// @param value            A value of type int64_t.
/// @param metricId         Metric ID.
///
/// @return TVCMI_Error_Success -           on success
/// @return TVCMI_Error_NoMemory -          no memory
/// @return TVCMI_Error_InvalidArguments -  wrong parameters
///
/// @see TVCMI_CreateMetricValueContainer()
/// @see TVCMI_DestroyMetricValueContainer()
/// @see TVCMI_AddBoolMetricValue()
/// @see TVCMI_AddDoubleMetricValue()
/// @see TVCMI_AddStringMetricValue()
/// @see TVCMI_AddEpochTimeStamp()
///
IOTTVCMI_API TVCMI_Error TVCMI_AddIntegerMetricValue(TVCMI_MetricValueContainer* container, int64_t value, const char* metricId);


///
/// Add a value of boolean type to the container object.
///
/// @param container        A valid TVCMI_MetricValueContainer object.
/// @param value            A value of boolean type. It can be 0 or 1.
/// @param metricId         Metric ID.
///
/// @return TVCMI_Error_Success -           on success
/// @return TVCMI_Error_NoMemory -          no memory
/// @return TVCMI_Error_InvalidArguments -  wrong parameters
///
/// @see TVCMI_CreateMetricValueContainer()
/// @see TVCMI_DestroyMetricValueContainer()
/// @see TVCMI_AddIntegerMetricValue()
/// @see TVCMI_AddDoubleMetricValue()
/// @see TVCMI_AddStringMetricValue()
/// @see TVCMI_AddEpochTimeStamp()
///
IOTTVCMI_API TVCMI_Error TVCMI_AddBoolMetricValue(TVCMI_MetricValueContainer* container, int value, const char* metricId);


///
/// Add a value of type double to the container object.
///
/// @param container        A valid TVCMI_MetricValueContainer object.
/// @param value            A value of type double.
/// @param metricId         Metric ID.
///
/// @return TVCMI_Error_Success -           on success
/// @return TVCMI_Error_NoMemory -          no memory
/// @return TVCMI_Error_InvalidArguments -  wrong parameters
///
/// @see TVCMI_CreateMetricValueContainer()
/// @see TVCMI_DestroyMetricValueContainer()
/// @see TVCMI_AddIntegerMetricValue()
/// @see TVCMI_AddBoolMetricValue()
/// @see TVCMI_AddStringMetricValue()
/// @see TVCMI_AddEpochTimeStamp()
///
IOTTVCMI_API TVCMI_Error TVCMI_AddDoubleMetricValue(TVCMI_MetricValueContainer* container, double value, const char* metricId);


///
/// Add a value of type string to the container object.
///
/// @param container        A valid TVCMI_MetricValueContainer object.
/// @param value            Null terminated string value.
/// @param metricId         Metric ID.
///
/// @return TVCMI_Error_Success -           on success
/// @return TVCMI_Error_NoMemory -          no memory
/// @return TVCMI_Error_InvalidArguments -  wrong parameters
///
/// @see TVCMI_CreateMetricValueContainer()
/// @see TVCMI_DestroyMetricValueContainer()
/// @see TVCMI_AddIntegerMetricValue()
/// @see TVCMI_AddBoolMetricValue()
/// @see TVCMI_AddDoubleMetricValue()
/// @see TVCMI_AddEpochTimeStamp()
///
IOTTVCMI_API TVCMI_Error TVCMI_AddStringMetricValue(TVCMI_MetricValueContainer* container, const char* value, const char* metricId);

///
/// Add a timestamp to the container object.
///
/// @param container        A valid TVCMI_MetricValueContainer object.
/// @param epochTimestamp   The timestamp has to be a UNIX Epoch timestamp. Time since 1.1.1970.
///
/// @return TVCMI_Error_Success -           on success
/// @return TVCMI_Error_InvalidArguments -  wrong parameters
///
/// @see TVCMI_CreateMetricValueContainer()
/// @see TVCMI_DestroyMetricValueContainer()
/// @see TVCMI_AddIntegerMetricValue()
/// @see TVCMI_AddBoolMetricValue()
/// @see TVCMI_AddDoubleMetricValue()
///
IOTTVCMI_API TVCMI_Error TVCMI_AddEpochTimeStamp(TVCMI_MetricValueContainer* container, uint64_t epochTimestamp);

///
/// Get information about sensors, including its
/// metrics. The Callback set by TVCMI_ListSensorsCallbackSet()
/// will return the information about sensors.
///
/// @param client                               A valid TVCMI_Client instance.
///
/// @return TVCMI_Error_Success -               on success
/// @return TVCMI_Error_InvalidArguments -      wrong parameters
/// @return TVCMI_Error_NoMemory -              out of memory
/// @return TVCMI_Error_NoConnectionToServer -  no connection with TeamViewer IOT Agent
/// @return TVCMI_Error_WrongProtocol -         internal protocol error communicating with
///                                             the TeamViewer IOT Agent
///
IOTTVCMI_API TVCMI_Error TVCMI_ListSensors(TVCMI_Client* client);


///
/// Get information about certain metric.
/// The callback set by TVCMI_DescribeMetricCallbackSet()
/// will return the information about the metric.
///
///
/// @param client                       A valid TVCMI_Client instance.
/// @param sensorId                     ID of a sensor.
/// @param metricId                     ID of a metric.
///
/// @return TVCMI_Error_Success -               on success
/// @return TVCMI_Error_InvalidArguments -      wrong parameters
/// @return TVCMI_Error_NoMemory -              out of memory
/// @return TVCMI_Error_NoConnectionToServer -  no connection with TeamViewer IOT Agent
/// @return TVCMI_Error_WrongProtocol -         internal protocol error communicating with
///                                             the TeamViewer IOT Agent
///
IOTTVCMI_API TVCMI_Error TVCMI_DescribeMetric(TVCMI_Client* client, const char* sensorId, const char* metricId);


///
/// Send error message about a client.
///
/// @param client               A valid TVCMI_Client instance.
/// @param error                Error code.
/// @param errMessage           Error message text.
///
/// @return TVCMI_Error_Success -               on success
/// @return TVCMI_Error_InvalidArguments -      wrong parameters
/// @return TVCMI_Error_NoMemory -              out of memory
/// @return TVCMI_Error_NoConnectionToServer -  no connection with TeamViewer IOT Agent
/// @return TVCMI_Error_WrongProtocol -         internal protocol error communicating with
///                                             the TeamViewer IOT Agent
///
IOTTVCMI_API TVCMI_Error TVCMI_ReportClientError(TVCMI_Client* client, int error, const char* errMessage);


///
/// Send error message about a sensor.
///
/// @param client               A valid TVCMI_Client instance.
/// @param sensorId             ID of a sensor.
/// @param error                Error code.
/// @param errMessage           Error message text.
///
/// @return TVCMI_Error_Success -               on success
/// @return TVCMI_Error_InvalidArguments -      wrong parameters
/// @return TVCMI_Error_NoMemory -              out of memory
/// @return TVCMI_Error_NoConnectionToServer -  no connection with TeamViewer IOT Agent
/// @return TVCMI_Error_WrongProtocol -         internal protocol error communicating with
///                                             the TeamViewer IOT Agent
///
IOTTVCMI_API TVCMI_Error TVCMI_ReportSensorError(
	TVCMI_Client* client,
	const char* sensorId,
	int error,
	const char* errMessage);


///
/// Send error message about a metric.
///
/// @param client               A valid TVCMI_Client instance.
/// @param sensorId             ID of a sensor.
/// @param metricId             ID of a metric.
/// @param error                Error code.
/// @param errMessage error     Message text.
///
/// @return TVCMI_Error_Success -               on success
/// @return TVCMI_Error_InvalidArguments -      wrong parameters
/// @return TVCMI_Error_NoMemory -              out of memory
/// @return TVCMI_Error_NoConnectionToServer -  no connection with TeamViewer IOT Agent
/// @return TVCMI_Error_WrongProtocol -         internal protocol error communicating with
///                                             the TeamViewer IOT Agent
///
IOTTVCMI_API TVCMI_Error TVCMI_ReportMetricError(
	TVCMI_Client* client,
	const char* sensorId,
	const char* metricId,
	int error,
	const char* errMessage);


///
/// Set TVCMI_OnProvisionClient callback to get an event about client being provisioned.
/// Do not make any other SDK function calls in this callback function.
///
/// @param client                   A valid TVCMI_Client instance.
/// @param TVCMI_OnProvisionClient  A callback function.
///
///     Callback Parameters:
///         client -            A valid TVCMI_Client instance.
///         callbackParameter - A pointer provided during creation of the client.
///         err -               Returned error code.
///
/// @return TVCMI_Error_Success -               on success
/// @return TVCMI_Error_InvalidArguments -      wrong parameters
/// @return TVCMI_Error_NoMemory -              no memory
/// @return TVCMI_Error_NoConnectionToServer -  no connection with TeamViewer IOT Agent
/// @return TVCMI_Error_WrongProtocol -         internal protocol error communicating with
///                                             the TeamViewer IOT Agent
/// @return TVCMI_Error_PayloadSize -           payload is too large
/// @return TVCMI_Error_WrongIdentifier -       internal identifier error
/// @return TVCMI_Error_ServerUnavailable -     TeamViewer IOT Agent is not available
///
IOTTVCMI_API TVCMI_Error TVCMI_ProvisionClientCallbackSet(
	TVCMI_Client* client,
	void (*TVCMI_OnProvisionClient)(
		TVCMI_Client* client,
		void* callbackParameter,
		TVCMI_Error err));


///
/// Set TVCMI_OnConnect callback to get an event about client being connected.
///
/// @param client               A valid TVCMI_Client instance.
/// @param TVCMI_OnConnect      A callback function.
///
///     Callback Parameters:
///         client -            The TVCMI_Client instance making the callback.
///         callbackParameter - A pointer provided during creation of the client.
///         err -               Returned error code.
///
/// @return TVCMI_Error_Success -           on success.
/// @return TVCMI_Error_InvalidArguments -  wrong parameters.
///
IOTTVCMI_API TVCMI_Error TVCMI_ConnectCallbackSet(
	TVCMI_Client* client,
	void (*TVCMI_OnConnect)(
		TVCMI_Client* client,
		void* callbackParameter,
		TVCMI_Error err));


///
/// Set TVCMI_OnDisconnect callback to get an event about client being disconnected.
///
/// @param client                   A valid TVCMI_Client instance.
/// @param TVCMI_OnDisconnect       A callback function.
///
///     Callback Parameters:
///         client -            The TVCMI_Client instance making the callback.
///         callbackParameter - A pointer provided during creation of the client.
///         err -               Returned error code.
///
/// @return TVCMI_Error_Success -           on success
/// @return TVCMI_Error_InvalidArguments -  wrong parameters
///
IOTTVCMI_API TVCMI_Error TVCMI_DisconnectCallbackSet(
	TVCMI_Client* client,
	void (*TVCMI_OnDisconnect)(
		TVCMI_Client* client,
		void* callbackParameter,
		TVCMI_Error err));


///
/// Set TVCMI_OnCreateSensor callback to get a response
/// about sensor creation.
///
/// @param client                   A valid TVCMI_Client instance.
/// @param TVCMI_OnCreateSensor     A callback function.
///
///     Callback Parameters:
///         client -            The TVCMI_Client instance making the callback.
///         callbackParameter - A pointer provided during creation of the client.
///         sensorName -        The name of the sensor. It will be destroyed automatically after callback returns.
///         sensorId -          The id of the sensor. It will be destroyed automatically after callback returns.
///         err -               Returned error code.
///
/// @return TVCMI_Error_Success -           on success
/// @return TVCMI_Error_InvalidArguments -  wrong parameters
///
IOTTVCMI_API TVCMI_Error TVCMI_CreateSensorCallbackSet(
	TVCMI_Client* client,
	void (*TVCMI_OnCreateSensor)(
		TVCMI_Client* client,
		void* callbackParameter,
		const char* sensorName,
		const char* sensorId,
		TVCMI_ApiError err));


///
/// Set TVCMI_OnUpdateSensor callback to get a response
/// about sensor meta data change.
///
/// @param client                   A valid TVCMI_Client instance.
/// @param TVCMI_OnUpdateSensor     A callback function.
///
///     Callback Parameters:
///         client -            The TVCMI_Client instance making the callback.
///         callbackParameter - A pointer provided during creation of the client.
///         sensorId -          The id of the sensor.
///         err -               Returned error code.
///
/// @return TVCMI_Error_Success -           on success
/// @return TVCMI_Error_InvalidArguments -  wrong parameters
///
IOTTVCMI_API TVCMI_Error TVCMI_UpdateSensorCallbackSet(
	TVCMI_Client* client,
	void (*TVCMI_OnUpdateSensor)(
		TVCMI_Client* client,
		void* callbackParameter,
		const char* sensorId,
		TVCMI_ApiError err));


///
/// Set TVCMI_OnDeleteSensor callback to get a response
/// about sensor deletion.
///
/// @param client                   A valid TVCMI_Client instance.
/// @param TVCMI_OnDeleteSensor     A callback function.
///
///     Callback Parameters:
///         client -            The TVCMI_Client instance making the callback.
///         callbackParameter - A pointer provided during creation of the client.
///         sensorId -          The id of the sensor.
///         err -               Returned error code.
///
/// @return TVCMI_Error_Success -           on success
/// @return TVCMI_Error_InvalidArguments -  wrong parameters
///
IOTTVCMI_API TVCMI_Error TVCMI_DeleteSensorCallbackSet(
	TVCMI_Client* client,
	void (*TVCMI_OnDeleteSensor)(
		TVCMI_Client* client,
		void* callbackParameter,
		const char* sensorId,
		TVCMI_ApiError err));


///
/// Set TVCMI_OnCreateMetrics callback to get a response about
/// metrics creation. createdMetrics must be destroyed by
/// call to function TVCMI_DestroyCreatedMetrics() after usage.
///
/// @param client                   A valid TVCMI_Client instance.
/// @param TVCMI_OnCreateMetrics    A callback function.
///
///     Callback Parameters:
///         client -                The TVCMI_Client instance making the callback.
///         callbackParameter -     A pointer provided during creation of the client.
///         createdMetrics -        A structure containing information about created
///                                 metrics. Created metrics must be destroyed by
///                                 function TVCMI_DestroyCreatedMetrics() after usage.
///         err -                   Returned error code.
///
/// @return TVCMI_Error_Success -           on success
/// @return TVCMI_Error_InvalidArguments -  wrong parameters
///
IOTTVCMI_API TVCMI_Error TVCMI_CreateMetricsCallbackSet(
	TVCMI_Client* client,
	void (*TVCMI_OnCreateMetrics)(
		TVCMI_Client* client,
		void* callbackParameter,
		TVCMI_CreatedMetrics* createdMetrics,
		TVCMI_ApiError err));

///
/// Set TVCMI_OnUpdateMetrics callback to get a response about
/// metrics update.
///
/// @param client                   A valid TVCMI_Client instance.
/// @param TVCMI_OnUpdateMetrics    A callback function.
///
///     Callback Parameters:
///         client -                The TVCMI_Client instance making the callback.
///         callbackParameter -     A pointer provided during creation of the client.
///         sensorId -              The id of the sensor.
///         err -                   Returned error code.
///
/// @return TVCMI_Error_Success -           on success
/// @return TVCMI_Error_InvalidArguments -  wrong parameters
///
IOTTVCMI_API TVCMI_Error TVCMI_UpdateMetricsCallbackSet(
	TVCMI_Client* client,
	void (*TVCMI_OnUpdateMetrics)(
		TVCMI_Client* client,
		void* callbackParameter,
		const char* sensorId,
		TVCMI_ApiError err));


///
/// Destroy created metrics returned by callback function TVCMI_OnCraeteMetrics().
///
/// @param createdMetrics            A pointer to structure containing created
///                                  metrics.
///
IOTTVCMI_API void TVCMI_DestroyCreatedMetrics(TVCMI_CreatedMetrics* createdMetrics);


///
/// Set TVCMI_OnDeleteMetrics callback to get errors about
/// metrics deletion.
///
/// @param client                   A valid TVCMI_Client instance.
/// @param TVCMI_OnDeleteMetrics    A callback function.
///
///     Callback Parameters:
///         client -            The TVCMI_Client instance making the callback.
///         callbackParameter - A pointer provided during creation of the client.
///         sensorId -          The id of the sensor.
///         err -               Returned error code.
///
/// @return TVCMI_Error_Success -           on success
/// @return TVCMI_Error_InvalidArguments -  wrong parameters
///
IOTTVCMI_API TVCMI_Error TVCMI_DeleteMetricsCallbackSet(
	TVCMI_Client* client,
	void (*TVCMI_OnDeleteMetrics)(
		TVCMI_Client* client,
		void* callbackParameter,
		const char* sensorId,
		TVCMI_ApiError err));


///
/// Set TVCMI_OnPushMetricValues callback to get errors about
/// metrics values update.
///
/// @param client                       A valid TVCMI_Client instance.
/// @param TVCMI_OnPushMetricValues     A callback function.
///
///     Callback Parameters:
///         client -            The TVCMI_Client instance making the callback.
///         callbackParameter - A pointer provided during creation of the client.
///         sensorId -          The id of the sensor.
///         err -               Returned error code.
///
/// @return TVCMI_Error_Success -           on success
/// @return TVCMI_Error_InvalidArguments -  wrong parameters
///
IOTTVCMI_API TVCMI_Error TVCMI_PushMetricValuesCallbackSet(
	TVCMI_Client* client,
	void (*TVCMI_OnPushMetricValues)(
		TVCMI_Client* client,
		void* callbackParameter,
		const char* sensorId,
		TVCMI_ApiError err));


///
/// Set TVCMI_OnListSensors callback to get a response
/// about the sensors.
///
/// @param client                           A valid TVCMI_Client instance.
/// @param TVCMI_OnListSensors              A callback function.
///
///     Callback Parameters:
///         client -            The TVCMI_Client instance making the callback.
///         sensorsInfo -       A pointer to a structure containing information about sensors.
///                             The pointer must be freed by function
///                             TVCMI_DestroySensorsInfo() after usage.
///         callbackParameter - A pointer provided during creation of the client.
///         err -               Returned error code.
///
/// @return TVCMI_Error_Success -           on success
/// @return TVCMI_Error_InvalidArguments -  wrong parameters
///
IOTTVCMI_API TVCMI_Error TVCMI_ListSensorsCallbackSet(
	TVCMI_Client* client,
	void (*TVCMI_OnListSensors)(
		TVCMI_Client* client,
		TVCMI_SensorsInfo* sensorsInfo,
		void* callbackParameter,
		TVCMI_ApiError err));


///
/// Destroy sensorsInfo returned by callback function TVCMI_OnListSensors().
///
/// @param sensorsInfo -        A structure containing sensors information.
///
IOTTVCMI_API void TVCMI_DestroySensorsInfo(TVCMI_SensorsInfo* sensorsInfo);


///
/// Set TVCMI_OnDescribeMetric callback to receive a
/// response about the metric.
///
/// @param client                   A valid TVCMI_Client instance.
/// @param TVCMI_OnDescribeMetric   A callback function.
///
///     Callback Parameters:
///         client -            The TVCMI_Client instance making the callback.
///         sensorId -          ID of a sensor to which the metric belongs to.
///         inventory -         A pointer to a structure containing inventory information.
///                             It must be deleted by call of function
///                             TVCMI_DestroyMetricInventory()
///         callbackParameter - A pointer provided during creation of the client.
///         err -               Returned error code.
///
/// @return TVCMI_Error_Success -           on success
/// @return TVCMI_Error_InvalidArguments -  wrong parameters
///
IOTTVCMI_API TVCMI_Error TVCMI_DescribeMetricCallbackSet(
	TVCMI_Client* client,
	void (*TVCMI_OnDescribeMetric)(
		TVCMI_Client* client,
		const char* sensorId,
		TVCMI_MetricMetadata* metricMetadata,
		void* callbackParameter,
		TVCMI_ApiError err));


///
/// Destroy metricMetadata returned by callback function
/// TVCMI_OnGetMetric().
///
/// @param metricMetadata -     A structure containing inventory information.
///
IOTTVCMI_API void TVCMI_DestroyMetricMetadata(TVCMI_MetricMetadata* metricMetadata);


///
/// Set TVCMI_OnLog callback to get library's internal info and error messages.
///
/// @param client           A valid TVCMI_Client instance.
/// @param TVCMI_OnLog      A callback function.
///
///     Callback Parameters:
///         client -            The TVCMI_Client instance making the callback.
///         logMessage -        The message text.
///         callbackParameter - A pointer provided during creation of the client.
///
/// @return TVCMI_Error_Success -           on success
/// @return TVCMI_Error_InvalidArguments -  wrong parameters
///
IOTTVCMI_API TVCMI_Error TVCMI_LogCallbackSet(
	TVCMI_Client* client,
	void (*TVCMI_OnLog)(
		TVCMI_Client* client,
		const char* logMessage,
		void* callbackParameter));


///
/// Return the client id.
///
/// @param client       A valid TVCMI_Client instance.
///
/// @return client Id -        on success
/// @return NULL -             on failure
///
IOTTVCMI_API const char* TVCMI_GetClientId(TVCMI_Client* client);


///
/// Check if the connection with TeamViewer IoT Agent is alive. If it works then you will get a
/// response callback set by function TVCMI_CheckConnectionCallbackSet().
///
/// @param client -         A valid TVCMI_Client instance.
///
/// @return client Id -     on success
/// @return NULL -          on failure
///
IOTTVCMI_API TVCMI_Error TVCMI_CheckConnection(TVCMI_Client* client);


///
/// Set TVCMI_OnCheckConnection callback to get a response from the TeamViewer IoT Agent.
/// If the connection is alive you will get the response.
///
/// @param client                           A valid TVCMI_Client instance.
/// @param TVCMI_OnCheckConnection          A callback function.
///
///     Callback Parameters:
///         client -            The TVCMI_Client instance making the callback.
///         callbackParameter - A pointer provided during creation of connetor.
///         err -               Returned error code.
///
/// @return TVCMI_Error_Success -           on success
/// @return TVCMI_Error_InvalidArguments -  wrong parameters
///
IOTTVCMI_API TVCMI_Error TVCMI_CheckConnectionCallbackSet(
	TVCMI_Client* client,
	void (*TVCMI_OnCheckConnection)(
		TVCMI_Client* client,
		void* callbackParameter,
		TVCMI_ApiError err));


///
/// Set TVCMI_OnCreateSensorJob callback to receive sensor create jobs.
///
/// @param client                   A valid TVCMI_Client instance.
/// @param TVCMI_OnCreateSensorJob  A callback function.
///
///     Callback Parameters:
///         client -            The TVCMI_Client instance making the callback.
///         sensorMetadata -    A pointer to a structure containing the sensor information.
///                             It must be deleted by call of function
///                             TVCMI_DestroySensorMetadata()
///         callbackParameter - A pointer provided during creation of the client.
///
/// @return TVCMI_Error_Success -           on success
/// @return TVCMI_Error_InvalidArguments -  wrong parameters
///
IOTTVCMI_API TVCMI_Error TVCMI_CreateSensorJobCallbackSet(
	TVCMI_Client* client,
	void (*TVCMI_OnCreateSensorJob)(
		TVCMI_Client* client,
		TVCMI_SensorMetadata* sensorMetadata,
		void* callbackParameter));


///
/// Destroy sensorMetadata returned by callback function TVCMI_OnCreateSensorJob().
///
/// @param sensorMetadata          A structure containing sensor information.
///
void TVCMI_DestroySensorMetadata(TVCMI_SensorMetadata* sensorMetadata);


///
/// Set TVCMI_OnDeleteSensorJob callback to receive sensor delete jobs.
///
/// @param client                   A valid TVCMI_Client instance.
/// @param TVCMI_OnDeleteSensorJob  A callback function.
///
///     Callback Parameters:
///         client -            The TVCMI_Client instance making the callback.
///         sensorId -          ID of a sensor to delete.
///         callbackParameter - A pointer provided during creation of the client.
///
/// @return TVCMI_Error_Success -           on success
/// @return TVCMI_Error_InvalidArguments -  wrong parameters
///
IOTTVCMI_API TVCMI_Error TVCMI_DeleteSensorJobCallbackSet(
	TVCMI_Client* client,
	void (*TVCMI_OnDeleteSensorJob)(
		TVCMI_Client* client,
		const char* sensorId,
		void* callbackParameter));


///
/// Delete the access to this client.
/// The callback function set by TVCMI_DeprovisionClientCallbackSet() will confirm that
/// deprovisioning is done.
///
/// @param client           A valid TVCMI_Client instance.
///
/// @return TVCMI_Error_Success -           on success
/// @return TVCMI_Error_InvalidArguments -  wrong parameters
///
IOTTVCMI_API TVCMI_Error TVCMI_DeprovisionClient(TVCMI_Client* client);


///
/// Set TVCMI_OnDeprovisionClient callback to know that the Deprovision is done.
///
/// @param client                           A valid TVCMI_Client instance.
/// @param TVCMI_OnDeprovisionClient        A callback function.
///
///     Callback Parameters:
///         client -            The TVCMI_Client instance making the callback.
///         callbackParameter - A pointer provided during creation of client.
///
IOTTVCMI_API TVCMI_Error TVCMI_DeprovisionClientCallbackSet(
	TVCMI_Client* client,
	void (*TVCMI_OnDeprovisionClient)(
		TVCMI_Client* client,
		void* callbackParameter));


#ifdef __cplusplus
}
#endif //__cplusplus

#endif //__TVCMI_CLIENT_H__

